/*************************************************************************************************
 *
 *   Copyright (c) Hilscher GmbH. All Rights Reserved.
 *
 **************************************************************************************************
 * @file CoEApplication.cpp
 *
 * @brief This file is contains functions for the example application which is interacting with CifX card 
 * through driver and packet API. 
 * 
 * @author R. Walter (basis: E. Ott)
 *
 */

#include "Debug.h"
#include "Common.h"
#include "CoE_Application.h"


/*************************************************************************************************
 * @brief This function registers all necessary SDO notify indications
 * 
 * @param ptCoEApp Pointer to CoE application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
static TLR_RESULT CoE_RegisterNotifications(COE_APPLICATION_T* ptCoEApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptCoEApp;

  ODV3_REGISTER_OBJECT_NOTIFY_REQ_T* ptRegNotifyReq = (ODV3_REGISTER_OBJECT_NOTIFY_REQ_T*)&tSendPkt;
  memset(ptRegNotifyReq, 0, sizeof(ptRegNotifyReq)); 

  /* fill packet header */
  ptRegNotifyReq->tHead.ulDest        = 0x20;
  ptRegNotifyReq->tHead.ulSrc         = 0x00;
  ptRegNotifyReq->tHead.ulDestId      = 0x00;
  ptRegNotifyReq->tHead.ulSrcId       = 0x00;
  ptRegNotifyReq->tHead.ulLen         = 3;
  ptRegNotifyReq->tHead.ulId          = 0x00;
  ptRegNotifyReq->tHead.ulSta         = 0x00;
  ptRegNotifyReq->tHead.ulCmd         = ODV3_REGISTER_OBJECT_NOTIFY_REQ;
  ptRegNotifyReq->tHead.ulExt         = 0x00;
  ptRegNotifyReq->tHead.ulRout        = 0x00;
   
  /* fill data part of packet */
  ptRegNotifyReq->tData.usIndex          = 0x6000;
  ptRegNotifyReq->tData.bIndicationFlags = ODV3_INDICATION_FLAGS_ON_WRITE;

  tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  return tResult;
}


/*************************************************************************************************
 * @brief This function unregisters from SDO notify indications
 * 
 * @param ptCoEApp Pointer to CoE application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
static TLR_RESULT CoE_UnegisterNotifications(COE_APPLICATION_T* ptCoEApp)
{  
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptCoEApp;

  ODV3_UNREGISTER_OBJECT_NOTIFY_REQ_T* ptUnRegNotifyReq = (ODV3_UNREGISTER_OBJECT_NOTIFY_REQ_T*)&tSendPkt;
  memset(ptUnRegNotifyReq, 0, sizeof(ptUnRegNotifyReq)); 

  /* fill packet header */
  ptUnRegNotifyReq->tHead.ulDest        = 0x20;
  ptUnRegNotifyReq->tHead.ulSrc         = 0x00;
  ptUnRegNotifyReq->tHead.ulDestId      = 0x00;
  ptUnRegNotifyReq->tHead.ulSrcId       = 0x00;
  ptUnRegNotifyReq->tHead.ulLen         = 2;
  ptUnRegNotifyReq->tHead.ulId          = 0x00;
  ptUnRegNotifyReq->tHead.ulSta         = 0x00;
  ptUnRegNotifyReq->tHead.ulCmd         = ODV3_UNREGISTER_OBJECT_NOTIFY_REQ;
  ptUnRegNotifyReq->tHead.ulExt         = 0x00;
  ptUnRegNotifyReq->tHead.ulRout        = 0x00;
   
  /* fill data part of packet */
  ptUnRegNotifyReq->tData.usIndex          = 0x6000;

  /* fire the request */
  tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  return tResult;
}


/*************************************************************************************************
 * @brief This function creates a simple variable object (no subobjects).
 * 
 * @param ptCoEApp Pointer to CoE application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
static TLR_RESULT CoE_CreateSimpleObject(COE_APPLICATION_T* ptCoEApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptCoEApp;

  ODV3_WRITE_OBJECT_REQ_T* ptWriteObjectReq = NULL;
  ODV3_SET_OBJECT_NAME_REQ_T* ptSetNameReq = NULL;
  char szName[] = "Demo Object";

  /* create object 0x6000 within object dictionary */
  /* variable, UINT32, no read / write restrictions, 1 subobject */
  ODV3_CREATE_OBJECT_REQ_T* ptCreateObjReq = (ODV3_CREATE_OBJECT_REQ_T*)&tSendPkt;
  
  
  memset(ptCreateObjReq, 0, sizeof(ODV3_CREATE_OBJECT_REQ_T));

  /* fill packet header */
  ptCreateObjReq->tHead.ulDest        = 0x20;
  ptCreateObjReq->tHead.ulSrc         = 0x00;
  ptCreateObjReq->tHead.ulDestId      = 0x00;
  ptCreateObjReq->tHead.ulSrcId       = 0x00;
  ptCreateObjReq->tHead.ulLen         = 20 + 0;
  ptCreateObjReq->tHead.ulId          = 0x00;
  ptCreateObjReq->tHead.ulSta         = 0x00;
  ptCreateObjReq->tHead.ulCmd         = ODV3_CREATE_OBJECT_REQ;
  ptCreateObjReq->tHead.ulExt         = 0x00;
  ptCreateObjReq->tHead.ulRout        = 0x00;

  /* fill data part of packet */
  ptCreateObjReq->tData.usIndex           = 0x6000;
  ptCreateObjReq->tData.bMaxNumOfSubObjs  = 0x00; 
  ptCreateObjReq->tData.bObjectCode       = 0x07; /* VAR    */
  ptCreateObjReq->tData.usAccessFlags     = 0x00;
  ptCreateObjReq->tData.usDatatype        = 0x0007; /* UINT32 */
  ptCreateObjReq->tData.usAccessRights    = ECAT_OD_ACCESS_ALL;
  ptCreateObjReq->tData.ulMaxFieldUnits   = 0x01; /* only one instance */
  ptCreateObjReq->tData.ulTotalDataBytes  = 0;
  ptCreateObjReq->tData.bValueInfo        = 0;

  /* fire the request */
  tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* download the initial value */
  ptWriteObjectReq = (ODV3_WRITE_OBJECT_REQ_T*)&tSendPkt;
  memset(ptWriteObjectReq, 0, sizeof(ODV3_WRITE_OBJECT_REQ_T));

  /* fill packet header */
  ptWriteObjectReq->tHead.ulDest            = 0x20;
  ptWriteObjectReq->tHead.ulSrc             = 0x00;
  ptWriteObjectReq->tHead.ulDestId          = 0x00;
  ptWriteObjectReq->tHead.ulSrcId           = 0x00;
  ptWriteObjectReq->tHead.ulLen             = 11 + 4;
  ptWriteObjectReq->tHead.ulId              = 0x00;
  ptWriteObjectReq->tHead.ulSta             = 0x00;
  ptWriteObjectReq->tHead.ulCmd             = ODV3_WRITE_OBJECT_REQ;
  ptWriteObjectReq->tHead.ulExt             = 0x00;
  ptWriteObjectReq->tHead.ulRout            = 0x00;

  /* fill data part of packet */   
  ptWriteObjectReq->tData.usIndex                = 0x6000;
  ptWriteObjectReq->tData.bSubIndex              = 0x00;
  ptWriteObjectReq->tData.ulTotalDataBytes       = ODV3_WRITE_OBJECT_REQ_TOTAL_DATA_BYTES_NOT_SPECIFIED; 

  /* always true in requests - only relevant within indications */
  ptWriteObjectReq->tData.fValidationIsImplicit  = TLR_TRUE;

  /* data value follows here */
  ptWriteObjectReq->tData.abData[0] = 0x00;
  ptWriteObjectReq->tData.abData[1] = 0x01;
  ptWriteObjectReq->tData.abData[2] = 0x02;
  ptWriteObjectReq->tData.abData[3] = 0x03;

  /* fire the request */
  tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* set the name of object 0x6000 */  
  ptSetNameReq = (ODV3_SET_OBJECT_NAME_REQ_T*)&tSendPkt;
  memset(ptSetNameReq, 0, sizeof(ODV3_SET_OBJECT_NAME_REQ_T));

  /* fill packet header */
  ptSetNameReq->tHead.ulDest            = 0x20;
  ptSetNameReq->tHead.ulSrc             = 0x00;
  ptSetNameReq->tHead.ulDestId          = 0x00;
  ptSetNameReq->tHead.ulSrcId           = 0x00;
  ptSetNameReq->tHead.ulLen             = 202;
  ptSetNameReq->tHead.ulId              = 0x00;
  ptSetNameReq->tHead.ulSta             = 0x00;
  ptSetNameReq->tHead.ulCmd             = ODV3_SET_OBJECT_NAME_REQ;
  ptSetNameReq->tHead.ulExt             = 0x00;
  ptSetNameReq->tHead.ulRout            = 0x00;

  /* fill data part of packet */        
  ptSetNameReq->tData.usIndex           = 0x6000;
  memcpy((char*)(&ptSetNameReq->tData.szName), szName, strlen(szName) + 1);

  /* fire the request */
  tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  return tResult;
}


/*************************************************************************************************
 * @brief This function creates a record object.
 * 
 * @param ptCoEApp Pointer to CoE application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
static TLR_RESULT CoE_CreateRecordObject(COE_APPLICATION_T* ptCoEApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  char szName[] = "Record Object";
  ODV3_SET_OBJECT_NAME_REQ_T* ptSetNameReq = NULL;
  ODV3_CREATE_OBJECT_REQ_T* ptCreateObjReq = NULL;
  ODV3_CREATE_SUBOBJECT_REQ_T* ptCreateSubObjReq = NULL;

  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptCoEApp;


  /* create object 0x6010 within object dictionary */
  ptCreateObjReq = (ODV3_CREATE_OBJECT_REQ_T*)&tSendPkt;
  memset(ptCreateObjReq, 0, sizeof(ODV3_CREATE_OBJECT_REQ_T));

  /* fill packet header */
  ptCreateObjReq->tHead.ulDest        = 0x20;
  ptCreateObjReq->tHead.ulSrc         = 0x00;
  ptCreateObjReq->tHead.ulDestId      = 0x00;
  ptCreateObjReq->tHead.ulSrcId       = 0x00;
  ptCreateObjReq->tHead.ulLen         = 20 + 5;
  ptCreateObjReq->tHead.ulId          = 0x00;
  ptCreateObjReq->tHead.ulSta         = 0x00;
  ptCreateObjReq->tHead.ulCmd         = ODV3_CREATE_OBJECT_REQ;
  ptCreateObjReq->tHead.ulExt         = 0x00;
  ptCreateObjReq->tHead.ulRout        = 0x00;

  /* fill data part of packet */
  ptCreateObjReq->tData.usIndex           = 0x6010;
  ptCreateObjReq->tData.bMaxNumOfSubObjs  = 0x03; 
  ptCreateObjReq->tData.bObjectCode       = 0x09; /* Record */
  ptCreateObjReq->tData.usAccessFlags     = ODV3_ACCESS_FLAGS_CREATE_SUBINDEX_0;
  ptCreateObjReq->tData.bValueInfo        = ODV3_VALUE_INFO_INITIAL_VALUE;       /* this packet includes the initial value of the object    */
  ptCreateObjReq->tData.usDatatype        = 0x40; /*0x40 manufacturer specific data type, see EtherCAT spec */
  ptCreateObjReq->tData.usAccessRights    = ECAT_OD_READ_ALL;
  ptCreateObjReq->tData.ulMaxFieldUnits   = 0x01; /* only one instance */
  ptCreateObjReq->tData.ulTotalDataBytes  = 5;
  ptCreateObjReq->tData.abData[0] = 1;
  ptCreateObjReq->tData.abData[1] = 0; 
  ptCreateObjReq->tData.abData[2] = 0; 
  ptCreateObjReq->tData.abData[3] = 0; 
  ptCreateObjReq->tData.abData[4] = 2; 

  /* fire the request */
  tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }


  /* set the name of object 0x6010 */  
  ptSetNameReq = (ODV3_SET_OBJECT_NAME_REQ_T*)&tSendPkt;
  memset(ptSetNameReq, 0, sizeof(ODV3_SET_OBJECT_NAME_REQ_T));

  /* fill packet header */
  ptSetNameReq->tHead.ulDest            = 0x20;
  ptSetNameReq->tHead.ulSrc             = 0x00;
  ptSetNameReq->tHead.ulDestId          = 0x00;
  ptSetNameReq->tHead.ulSrcId           = 0x00;
  ptSetNameReq->tHead.ulLen             = 202;
  ptSetNameReq->tHead.ulId              = 0x00;
  ptSetNameReq->tHead.ulSta             = 0x00;
  ptSetNameReq->tHead.ulCmd             = ODV3_SET_OBJECT_NAME_REQ;
  ptSetNameReq->tHead.ulExt             = 0x00;
  ptSetNameReq->tHead.ulRout            = 0x00;

  /* fill data part of packet */        
  ptSetNameReq->tData.usIndex           = 0x6010;
  memcpy((char*)(&ptSetNameReq->tData.szName), szName, strlen(szName) + 1);

  /* fire the request */
  tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }


  /* subindex 0x01: variable UINT32 */
  ptCreateSubObjReq = (ODV3_CREATE_SUBOBJECT_REQ_T*)&tSendPkt;	
  memset(ptCreateSubObjReq, 0, sizeof(ptCreateSubObjReq)); 

  /* fill packet header */
  ptCreateSubObjReq->tHead.ulDest        = 0x20;
  ptCreateSubObjReq->tHead.ulSrc         = 0x00;
  ptCreateSubObjReq->tHead.ulDestId      = 0x00;
  ptCreateSubObjReq->tHead.ulSrcId       = 0x00;
  ptCreateSubObjReq->tHead.ulLen         = 17;
  ptCreateSubObjReq->tHead.ulId          = 0x00;
  ptCreateSubObjReq->tHead.ulSta         = 0x00;
  ptCreateSubObjReq->tHead.ulCmd         = ODV3_CREATE_SUBOBJECT_REQ;
  ptCreateSubObjReq->tHead.ulExt         = 0x00;
  ptCreateSubObjReq->tHead.ulRout        = 0x00;

  /* fill data part of packet */
  ptCreateSubObjReq->tData.usIndex           = 0x6010; 
  ptCreateSubObjReq->tData.bSubIndex         = 0x01;
  ptCreateSubObjReq->tData.bValueInfo        = 0x00;
  ptCreateSubObjReq->tData.bIndicationFlags  = 0x00;
  ptCreateSubObjReq->tData.usAccessRights    = ECAT_OD_ACCESS_ALL;
  ptCreateSubObjReq->tData.usDatatype        = 0x07;
  ptCreateSubObjReq->tData.ulMaxFieldUnits   = 0x01;
  ptCreateSubObjReq->tData.ulTotalDataBytes  = 0x00;

  /* fire the request */
  tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* subindex 0x02: variable UINT8 */
  memset(ptCreateSubObjReq, 0, sizeof(ptCreateSubObjReq)); 

  /* fill packet header */
  ptCreateSubObjReq->tHead.ulDest        = 0x20;
  ptCreateSubObjReq->tHead.ulSrc         = 0x00;
  ptCreateSubObjReq->tHead.ulDestId      = 0x00;
  ptCreateSubObjReq->tHead.ulSrcId       = 0x00;
  ptCreateSubObjReq->tHead.ulLen         = 17;
  ptCreateSubObjReq->tHead.ulId          = 0x00;
  ptCreateSubObjReq->tHead.ulSta         = 0x00;
  ptCreateSubObjReq->tHead.ulCmd         = ODV3_CREATE_SUBOBJECT_REQ;
  ptCreateSubObjReq->tHead.ulExt         = 0x00;
  ptCreateSubObjReq->tHead.ulRout        = 0x00;

  /* fill data part of packet */
  ptCreateSubObjReq->tData.usIndex           = 0x6010; 
  ptCreateSubObjReq->tData.bSubIndex         = 0x02;
  ptCreateSubObjReq->tData.bValueInfo        = 0x00;
  ptCreateSubObjReq->tData.bIndicationFlags  = 0x00;
  ptCreateSubObjReq->tData.usAccessRights    = ECAT_OD_ACCESS_ALL;
  ptCreateSubObjReq->tData.usDatatype        = 0x05;
  ptCreateSubObjReq->tData.ulMaxFieldUnits   = 0x01;
  ptCreateSubObjReq->tData.ulTotalDataBytes  = 0x00;

  /* fire the request */
  tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  return tResult;
}


/*************************************************************************************************
 * @brief This function creates a array object.
 * 
 * @param ptCoEApp Pointer to CoE application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
static TLR_RESULT CoE_CreateArrayObject(COE_APPLICATION_T* ptCoEApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  char szName[] = "Array Object";

  ODV3_SET_OBJECT_NAME_REQ_T* ptSetNameReq;
  TLR_UINT32 uiCnt;

  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptCoEApp;


  /* create object 0x6000 within object dictionary */
  /* variable, UINT32, no read / write restrictions, 1 subobject */
  /* demonstrating how to pass initial value within this packet*/
  ODV3_CREATE_OBJECT_REQ_T* ptCreateObjReq = (ODV3_CREATE_OBJECT_REQ_T*)&tSendPkt;
  memset(ptCreateObjReq, 0, sizeof(ODV3_CREATE_OBJECT_REQ_T));

  /* fill packet header */
  ptCreateObjReq->tHead.ulDest        = 0x20;
  ptCreateObjReq->tHead.ulSrc         = 0x00;
  ptCreateObjReq->tHead.ulDestId      = 0x00;
  ptCreateObjReq->tHead.ulSrcId       = 0x00;
  ptCreateObjReq->tHead.ulLen         = 20 + 5;
  ptCreateObjReq->tHead.ulId          = 0x00;
  ptCreateObjReq->tHead.ulSta         = 0x00;
  ptCreateObjReq->tHead.ulCmd         = ODV3_CREATE_OBJECT_REQ;
  ptCreateObjReq->tHead.ulExt         = 0x00;
  ptCreateObjReq->tHead.ulRout        = 0x00;

  /* fill data part of packet */
  ptCreateObjReq->tData.usIndex           = 0x6020;
  ptCreateObjReq->tData.bMaxNumOfSubObjs  = 10; 
  ptCreateObjReq->tData.bObjectCode       = 0x08;   /* ARRAY    */
  ptCreateObjReq->tData.usDatatype        = 0x0007; /* UINT32 */
  ptCreateObjReq->tData.usAccessFlags     = ODV3_ACCESS_FLAGS_CREATE_SUBINDEX_0; /* this flag triggers automatic generation of subindex 0 */
  ptCreateObjReq->tData.bValueInfo        = ODV3_VALUE_INFO_INITIAL_VALUE;       /* this packet includes the initial value of the object    */
  ptCreateObjReq->tData.bIndicationFlags  = 0x00;
  ptCreateObjReq->tData.usAccessRights    = ECAT_OD_READ_ALL;
  ptCreateObjReq->tData.ulMaxFieldUnits   = 0x01;
  ptCreateObjReq->tData.ulTotalDataBytes  = 0x05;
  ptCreateObjReq->tData.abData[0] = 1;
  ptCreateObjReq->tData.abData[1] = 0; 
  ptCreateObjReq->tData.abData[2] = 0; 
  ptCreateObjReq->tData.abData[3] = 0; 
  ptCreateObjReq->tData.abData[4] = 10; 

  /* fire the request */
  tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* set the name of object 0x7000 */ 
  ptSetNameReq = (ODV3_SET_OBJECT_NAME_REQ_T*)&tSendPkt;
  memset(ptSetNameReq, 0, sizeof(ODV3_SET_OBJECT_NAME_REQ_T));

  /* fill packet header */
  ptSetNameReq->tHead.ulDest            = 0x20;
  ptSetNameReq->tHead.ulSrc             = 0x00;
  ptSetNameReq->tHead.ulDestId          = 0x00;
  ptSetNameReq->tHead.ulSrcId           = 0x00;
  ptSetNameReq->tHead.ulLen             = 202;
  ptSetNameReq->tHead.ulId              = 0x00;
  ptSetNameReq->tHead.ulSta             = 0x00;
  ptSetNameReq->tHead.ulCmd             = ODV3_SET_OBJECT_NAME_REQ;
  ptSetNameReq->tHead.ulExt             = 0x00;
  ptSetNameReq->tHead.ulRout            = 0x00;

  /* fill data part of packet */        
  ptSetNameReq->tData.usIndex           = 0x6020;
  memcpy((char*)(&ptSetNameReq->tData.szName), szName, strlen(szName) + 1);

  /* fire the request */
  tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* create subobjects (1-10) */
  uiCnt = 0;
  for (; uiCnt < 10; uiCnt++)
  {
  	ODV3_CREATE_SUBOBJECT_REQ_T* ptCreateSubObjReq = (ODV3_CREATE_SUBOBJECT_REQ_T*)&tSendPkt;
	  memset(ptCreateSubObjReq, 0, sizeof(ptCreateSubObjReq)); 

    /* fill packet header */
    ptCreateSubObjReq->tHead.ulDest        = 0x20;
    ptCreateSubObjReq->tHead.ulSrc         = 0x00;
    ptCreateSubObjReq->tHead.ulDestId      = 0x00;
    ptCreateSubObjReq->tHead.ulSrcId       = 0x00;
    ptCreateSubObjReq->tHead.ulLen         = 17;
    ptCreateSubObjReq->tHead.ulId          = 0x00;
    ptCreateSubObjReq->tHead.ulSta         = 0x00;
    ptCreateSubObjReq->tHead.ulCmd         = ODV3_CREATE_SUBOBJECT_REQ;
    ptCreateSubObjReq->tHead.ulExt         = 0x00;
    ptCreateSubObjReq->tHead.ulRout        = 0x00;

    /* fill data part of packet */
    ptCreateSubObjReq->tData.usIndex           = 0x6020; 
    ptCreateSubObjReq->tData.bSubIndex         = uiCnt + 1;
    ptCreateSubObjReq->tData.bValueInfo        = 0x00;
    ptCreateSubObjReq->tData.bIndicationFlags  = 0x00;
    ptCreateSubObjReq->tData.usAccessRights    = ECAT_OD_ACCESS_ALL;
    ptCreateSubObjReq->tData.usDatatype        = 0x07;
    ptCreateSubObjReq->tData.ulMaxFieldUnits   = 0x01;
    ptCreateSubObjReq->tData.ulTotalDataBytes  = 0x00;


    /* fire the request */
    tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
    if (TLR_S_OK != tResult)
    {
      return tResult;
    }

    /* it is an array, so we do NOT have to name the subobjects */
  }

  return tResult;
}

  
/*************************************************************************************************
 * @brief This function creates all objects within the CoE stack.
 * 
 * @param ptCoEApp Pointer to CoE application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
static TLR_RESULT CoE_CreateObjects(COE_APPLICATION_T* ptCoEApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptCoEApp;


  /* create simple variable object */
  tResult = CoE_CreateSimpleObject(ptCoEApp);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* create record object */
  tResult = CoE_CreateRecordObject(ptCoEApp);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* create array object */
  tResult = CoE_CreateArrayObject(ptCoEApp);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  return tResult;
}
      

/*************************************************************************************************
 * @brief This function handles all packets related to Confirmed AlStatus Services.
 * It receives the state change requests as indication 
 * and triggers the state changes via sending a request packet the stack
 * 
 * @param ptCoEApp          Pointer to application data.
 * @param ptAlControlInd    Pointer to CifX packet.
 *
 */
static void CoE_HandleAlControlChangedRequest(COE_APPLICATION_T*  ptCoEApp, ECAT_ESM_ALCONTROL_CHANGED_IND_T* ptAlControlInd)
{
  /* variable for result values */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* convert to common application */
  APPLICATION_T* ptApp = (APPLICATION_T*)ptCoEApp;

  /* casts to specific packets */
  ECAT_ESM_SET_ALSTATUS_REQ_T* ptAlStatusReq  = (ECAT_ESM_SET_ALSTATUS_REQ_T*) &tSendPkt;  


  /* we allow all requested state changes */
  switch(ptAlControlInd->tData.tAlControl.uState)
  {
    /* Init->PreOp */
    case ECAT_AL_STATE_PRE_OPERATIONAL:

      /* fill packet header */
      ptAlStatusReq->tHead.ulCmd     = ECAT_ESM_SET_ALSTATUS_REQ;
      ptAlStatusReq->tHead.ulSta     = TLR_S_OK;
      ptAlStatusReq->tHead.ulLen     = sizeof(ptAlStatusReq->tData);
      ptAlStatusReq->tHead.ulExt     = 0;
      ptAlStatusReq->tHead.ulSrc     = 0;
      ptAlStatusReq->tHead.ulSrcId   = 0;
      ptAlStatusReq->tHead.ulDest    = 0x20;
      ptAlStatusReq->tHead.ulDestId  = 0;
      ptAlStatusReq->tHead.ulRout    = 0;
    
      /* fill data part of packet */
      ptAlStatusReq->tData.bAlStatus      = ECAT_AL_STATE_PRE_OPERATIONAL;
      ptAlStatusReq->tData.bErrorLedState = 0; /* TODO: handle LED if necessary */
      ptAlStatusReq->tData.usAlStatusCode = 0;

      /* send packet */
      tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
      break;

    /* PreOp->SafeOp */
    case ECAT_AL_STATE_SAFE_OPERATIONAL:

      /* fill packet header */
      ptAlStatusReq->tHead.ulCmd     = ECAT_ESM_SET_ALSTATUS_REQ;
      ptAlStatusReq->tHead.ulSta     = TLR_S_OK;
      ptAlStatusReq->tHead.ulLen     = sizeof(ptAlStatusReq->tData);
      ptAlStatusReq->tHead.ulExt     = 0;
      ptAlStatusReq->tHead.ulSrc     = 0;
      ptAlStatusReq->tHead.ulSrcId   = 0;
      ptAlStatusReq->tHead.ulDest    = 0x20;
      ptAlStatusReq->tHead.ulDestId  = 0;
      ptAlStatusReq->tHead.ulRout    = 0;

      /* fill data part of packet */
      ptAlStatusReq->tData.bAlStatus      = ECAT_AL_STATE_SAFE_OPERATIONAL;
      ptAlStatusReq->tData.bErrorLedState = 0; /* TODO: handle LED if necessary */
      ptAlStatusReq->tData.usAlStatusCode = 0;

      /* send packet */
      tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
      break;

    /* SafeOp->Op */
    case ECAT_AL_STATE_OPERATIONAL:

      /* fill packet header */
      ptAlStatusReq->tHead.ulCmd     = ECAT_ESM_SET_ALSTATUS_REQ;
      ptAlStatusReq->tHead.ulSta     = TLR_S_OK;
      ptAlStatusReq->tHead.ulLen     = sizeof(ptAlStatusReq->tData);
      ptAlStatusReq->tHead.ulExt     = 0;
      ptAlStatusReq->tHead.ulSrc     = 0;
      ptAlStatusReq->tHead.ulSrcId   = 0;
      ptAlStatusReq->tHead.ulDest    = 0x20;
      ptAlStatusReq->tHead.ulDestId  = 0;
      ptAlStatusReq->tHead.ulRout    = 0;

      /* fill data part of packet */
      ptAlStatusReq->tData.bAlStatus      = ECAT_AL_STATE_OPERATIONAL;
      ptAlStatusReq->tData.bErrorLedState = 0; /* TODO: handle LED if necessary */
      ptAlStatusReq->tData.usAlStatusCode = 0;

      /* send packet */
      tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
      break;

    /* unknown state */
    default:
      tResult = TLR_E_FAIL;
      break;
  }

  /* recycle the ind, generate res and reply it */
  ptAlControlInd->tHead.ulCmd |= 0x01;
  ptAlControlInd->tHead.ulLen  = 0x00;
  ptAlControlInd->tHead.ulSta  = tResult;

  /* send packet */
  xChannelPutPacket(ptCoEApp->tCommon.hChannel, (CIFX_PACKET*)ptAlControlInd, ptCoEApp->tCommon.ulTimeout);
}


/*************************************************************************************************
 * @brief This function handles indications from the stack which indicate 
 * that a change of al status has happened.
 * 
 * @param ptCoEApp          Pointer to CoE application data.
 * @param ptAlStatusInd     Pointer to CifX packet.
 *
 */
static void CoE_HandleAlStatusChangedIndication(COE_APPLICATION_T*  ptCoEApp, ECAT_ESM_ALSTATUS_CHANGED_IND_T* ptAlStatusInd)
{
  /* do not care, just cast and respond */
  /* currently no further actions needed */
  ECAT_ESM_ALSTATUS_CHANGED_RES_T* ptAlStatusRes = (ECAT_ESM_ALSTATUS_CHANGED_RES_T*)ptAlStatusInd;

  /* fill packet header */
  ptAlStatusRes->tHead.ulCmd  |= 0x01; 
  ptAlStatusRes->tHead.ulSta  = TLR_S_OK;
  ptAlStatusRes->tHead.ulLen  = 0;

  /* return to sender */
  xChannelPutPacket(ptCoEApp->tCommon.hChannel, (CIFX_PACKET*)ptAlStatusRes, ptCoEApp->tCommon.ulTimeout);
}


/*************************************************************************************************
 * @brief This function handles all SDO write indications.
 * 
 * @param ptCoEApp          Pointer to application data.
 * @param ptPacket          Pointer to CifX packet.
 *
 */
static void CoE_HandleWriteIndication(COE_APPLICATION_T*  ptCoEApp, ODV3_WRITE_OBJECT_IND_T* ptPacket)
{
  TLR_UINT32 ulValue = 0x00;

  switch(ptPacket->tData.usIndex)
  {
    /********************************
     * SDO 0x6000:  Demo Object
     */    
    case 0x6000:
      if (ptPacket->tData.bSubIndex == 0x00)
      {
        if ((ptPacket->tData.ulTotalDataBytes ==  4) && (ptPacket->tData.fValidationIsImplicit != TLR_FALSE))
        {
          /* build the value */
          ulValue |= ptPacket->tData.abData[0] << 0;
          ulValue |= ptPacket->tData.abData[1] << 8;
          ulValue |= ptPacket->tData.abData[2] << 16;
          ulValue |= ptPacket->tData.abData[3] << 24;
    
          /* prompt the value */    
          DEBUG("\n");
          DEBUG("Object 0x6000 has been written.\n");
          DEBUG("New Value: 0x%x\n", ulValue );
    
          /* build the response and send it */
          ptPacket->tHead.ulSta = TLR_S_OK;
          ptPacket->tHead.ulCmd |= 0x01;
          xChannelPutPacket(ptCoEApp->tCommon.hChannel, (CIFX_PACKET*)ptPacket, ptCoEApp->tCommon.ulTimeout);  
        }
        else
        {
          /* invalid data length or more than one application registered */

          /* build the response and send it */
          ptPacket->tHead.ulSta = TLR_E_FAIL;
          ptPacket->tHead.ulCmd |= 0x01;
    
          xChannelPutPacket(ptCoEApp->tCommon.hChannel, (CIFX_PACKET*)ptPacket, ptCoEApp->tCommon.ulTimeout);  
        }
      }
      else
      {
        /* invalid subindex */

        /* build the response and send it */
        ptPacket->tHead.ulSta = TLR_E_FAIL;
        ptPacket->tHead.ulCmd |= 0x01;
  
        xChannelPutPacket(ptCoEApp->tCommon.hChannel, (CIFX_PACKET*)ptPacket, ptCoEApp->tCommon.ulTimeout);  
      }

      break;
    
    default:
      /* invalid object index */

      /* build the response and send it */
      ptPacket->tHead.ulSta = TLR_E_FAIL;
      ptPacket->tHead.ulCmd |= 0x01;

      xChannelPutPacket(ptCoEApp->tCommon.hChannel, (CIFX_PACKET*)ptPacket, ptCoEApp->tCommon.ulTimeout);  
      break;
  }

}


/*************************************************************************************************
 * @brief This function handles all SDO read indications.
 * 
 * @param ptCoEApp          Pointer to application data.
 * @param ptPacket          Pointer to CifX packet.
 *
 */            
static void CoE_HandleReadIndication(COE_APPLICATION_T*  ptCoEApp, ODV3_READ_OBJECT_IND_T* ptPacket)
{
  /* we expect no read indications */
  switch(ptPacket->tData.usIndex)
  {
    case 0x6000:
      /* implement handling here if necessary */
      /* break; */

    default:
      /* build the response and send it */
      ptPacket->tHead.ulSta = TLR_E_FAIL;
      ptPacket->tHead.ulCmd |= 0x01;

      xChannelPutPacket(ptCoEApp->tCommon.hChannel, (CIFX_PACKET*)ptPacket, ptCoEApp->tCommon.ulTimeout);
      break;
  }
}


/*************************************************************************************************
 * @brief This function prompts a debug message at application startup.
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_PromptIntro(APPLICATION_T* ptApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  DEBUG("\n");
  DEBUG("**********************************************************\n");
  DEBUG("*                                                        *\n");
  DEBUG("*   Basic sample application for EtherCAT CoE            *\n");
  DEBUG("*                                                        *\n");
  DEBUG("*   Copyright (c) Hilscher GmbH. All Rights Reserved.    *\n");
  DEBUG("*                                                        *\n");
  DEBUG("**********************************************************\n");
  DEBUG("\n");

  return tResult;
}


/*************************************************************************************************
 * @brief This method builds a V2 configuration request packet.
 * 
 * @param CIFX_PACKET* ptPacket pointer to a CIFX_PACKET structur.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_BuildConfigurationReqV2 (CIFX_PACKET* ptPacket)
{
  TLR_RESULT tResult = TLR_S_OK;
  ECAT_DPM_SET_CONFIGURATION_REQ_T* ptConfigReq = (ECAT_DPM_SET_CONFIGURATION_REQ_T*)ptPacket;


  memset(ptConfigReq, 0, sizeof(ptConfigReq));

  /* fill packet header */
  ptConfigReq->tHead.ulDest = 0x20;                               /* Destination of packet                   */
  ptConfigReq->tHead.ulSrc = 0x00;                                /* Source of packet, process queue         */
  ptConfigReq->tHead.ulLen = 89;                                  /* Length of packet data without header    */
  ptConfigReq->tHead.ulCmd = ECAT_DPM_SET_CONFIGURATION_REQ;      /* Packet command                          */
  ptConfigReq->tHead.ulSta = 0;                                   /* Status code of operation                */
  ptConfigReq->tHead.ulExt = 0;                                   /* Extension                               */

  /* fill data part of packet */
  ptConfigReq->tData.ulSystemFlags = 0x01;                        /* Application controlled startup          */
  ptConfigReq->tData.ulWatchdogTime = 1000;                       /* Watchdog time                           */
  ptConfigReq->tData.ulVendorId = 0xE0000044;                     /* Vendor Id                               */
  ptConfigReq->tData.ulProductCode = 0x0000001E;                  /* Product code                            */
  ptConfigReq->tData.ulRevisionNumber = 0x00020004;               /* Revision number                         */
  ptConfigReq->tData.ulSerialNumber = 0x12345678;                 /* Serial number                           */
  ptConfigReq->tData.ulProcessDataOutputSize = MAX_RTD_SIZE;      /* Process Data Output Size                */
  ptConfigReq->tData.ulProcessDataInputSize = MAX_RTD_SIZE;       /* Process Data Input Size                 */

  ptConfigReq->tData.ulStackConfigurationFlags = 
    MSK_ECAT_DPM_SET_CONFIG_STACK_CFG_CLEAR_APPLICATION_OBJECTS;  /* Stack Configuration Flags               */
  ptConfigReq->tData.ulSIIConfigurationFlags = 0;                 /* SII Configuration Flags                 */
  ptConfigReq->tData.bSyncPdiConfig = 0xCC;                       /* Sync PDI configuration                  */
  ptConfigReq->tData.usSyncImpulseLength = 1000;                  /* Sync impulse length (in units of 10 ns) */
  ptConfigReq->tData.ulDeviceType = 0;                            /* Device type in object 0x1000            */

  return tResult;
}


/*************************************************************************************************
 * @brief This method builds a V4 configuration request packet.
 * 
 * @param CIFX_PACKET* ptPacket pointer to a CIFX_PACKET structur.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_BuildConfigurationReqV4 (CIFX_PACKET* ptPacket)
{
  TLR_RESULT tResult = TLR_S_OK;
  ECAT_SET_CONFIG_REQ_T* ptConfigReq = (ECAT_SET_CONFIG_REQ_T*)ptPacket;
  
   
  memset(ptConfigReq, 0, sizeof(ptConfigReq));

  /* fill packet header */
  ptConfigReq->tHead.ulDest = 0x20;                                         /* Destination of packet                   */
  ptConfigReq->tHead.ulSrc = 0x00;                                          /* Source of packet, process queue         */
  ptConfigReq->tHead.ulLen = sizeof(ECAT_SET_CONFIG_REQ_DATA_T);            /* Length of packet data without header    */
  ptConfigReq->tHead.ulCmd = ECAT_SET_CONFIG_REQ;                           /* Packet command                          */
  ptConfigReq->tHead.ulSta = 0;                                             /* Status code of operation                */
  ptConfigReq->tHead.ulExt = 0;                                             /* Extension                               */

  /* fill data part of packet */
  ptConfigReq->tData.tBasicCfg.ulSystemFlags = 0x01;                        /* Application controlled startup          */
  ptConfigReq->tData.tBasicCfg.ulWatchdogTime = 1000;                       /* Watchdog time                           */
  ptConfigReq->tData.tBasicCfg.ulVendorId = 0xE0000044;                     /* Vendor Id                               */
  ptConfigReq->tData.tBasicCfg.ulProductCode = 0x0000001E;                  /* Product code                            */
  ptConfigReq->tData.tBasicCfg.ulRevisionNumber = 0x00020004;               /* Revision number                         */
  ptConfigReq->tData.tBasicCfg.ulSerialNumber = 0x12345678;                 /* Serial number                           */
  ptConfigReq->tData.tBasicCfg.ulProcessDataOutputSize = MAX_RTD_SIZE;      /* Process Data Output Size                */
  ptConfigReq->tData.tBasicCfg.ulProcessDataInputSize = MAX_RTD_SIZE;       /* Process Data Input Size                 */
  ptConfigReq->tData.tBasicCfg.ulComponentInitialization = 0;               /* No components in use                    */
  ptConfigReq->tData.tBasicCfg.ulExtensionNumber = 0;                       /* No extension number in use              */

  return tResult;
}

/*************************************************************************************************
 * @brief This function creates all resources for the application. 
 * It allocates memory for the application data and returns a pointer to it.
 * App_FreeResources() must be called in order to free resources again.
 * 
 * @param pptApp    Pointer to Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_CreateResources(APPLICATION_T** pptApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  COE_APPLICATION_T* ptCoEApp = NULL;
  APPLICATION_T* ptApp = NULL;


  DEBUG("Allocating resources...\n");

  /* at the end of the function we will return the pointer */
  *pptApp = NULL;  
  
  /* allocate memory for application data */
  ptCoEApp = (COE_APPLICATION_T*)malloc(sizeof(COE_APPLICATION_T));
  memset(ptCoEApp, 0, sizeof(COE_APPLICATION_T));
  
  /* convert to common application */
  ptApp = (APPLICATION_T*)ptCoEApp;

  /* allocate memory for read / write buffers */
  ptCoEApp->tCommon.ulReadBufferSize  = MAX_RTD_SIZE * sizeof(TLR_UINT8);
  ptCoEApp->tCommon.ulWriteBufferSize = MAX_RTD_SIZE * sizeof(TLR_UINT8);

  ptCoEApp->tCommon.pabReadBuffer  =  (TLR_UINT8*) malloc(ptCoEApp->tCommon.ulReadBufferSize);
  ptCoEApp->tCommon.pabWriteBuffer =  (TLR_UINT8*) malloc(ptCoEApp->tCommon.ulWriteBufferSize);

  /* initialize the read and write buffer with zero */
  memset(ptApp->tCommon.pabReadBuffer,  0, ptApp->tCommon.ulReadBufferSize);
  memset(ptApp->tCommon.pabWriteBuffer, 0, ptApp->tCommon.ulWriteBufferSize);

  /* return the pointer */
  *pptApp = ptApp;

  DEBUG("Successful.\n");
  return tResult;
}

            
/*************************************************************************************************
 * @brief This function frees all resources created by App_CreateResources(). 
 * 
 * @param pptApp    Pointer to Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_FreeResources(APPLICATION_T* ptApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  COE_APPLICATION_T* ptCoEApp = NULL;

  /* convert to specific application */
  ptCoEApp = (COE_APPLICATION_T*)ptApp;


  DEBUG("Free resources...\n");

  /* free buffer resources */
  free(ptCoEApp->tCommon.pabReadBuffer);
  free(ptCoEApp->tCommon.pabWriteBuffer);

  /* free application data container */
  free(ptApp);

  DEBUG("Successful.\n");
  return tResult;
}


/*************************************************************************************************
 * @brief This function initializes the application. 
 * Objects will be creates and services will be registered inside.
 * App_Finalize() must be called in order to achieve a friendly shutdown of application.
 * 
 * @param ptApp    Pointer to Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Initialize(APPLICATION_T* ptApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* convert to specific application */
  COE_APPLICATION_T* ptCoEApp = (COE_APPLICATION_T*)ptApp;


  DEBUG("Initializing application...\n");

  /* create all user objects */
  tResult = CoE_CreateObjects(ptCoEApp);
  if(TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* register on EcatEsm task (indications for AL status) */
  tResult = App_SendRecvEmptyPkt(ptApp, RCX_REGISTER_APP_REQ);
  if(TLR_S_OK != tResult)
  {
    return tResult;
  }  
  
  /* register on EcatEsm task for indications if AL control changes 
   * (we want to get involved into the bootup of the slave stack)
   */
  tResult = App_SendRecvEmptyPkt(ptApp, ECAT_ESM_REGISTER_FOR_ALCONTROL_INDICATIONS_REQ);
  if(TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* register for read/write notifications */
  tResult = CoE_RegisterNotifications(ptCoEApp);
  if(TLR_S_OK != tResult)
  {
    return tResult;
  }

  return tResult;
}


/*************************************************************************************************
 * @brief This method finalizes the application. 
 * It returns handles, aso.
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Finalize(APPLICATION_T* ptApp)
 {
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* packets for sending and receiving */
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  /* convert to specific application */
  COE_APPLICATION_T* ptCoEApp = (COE_APPLICATION_T*)ptApp;

  
  DEBUG("Shutdown application...\n");

  /* at first we empty the queue, than we start with the shutdown sequence below */
  while(TLR_S_OK == (tResult = xChannelGetPacket(ptApp->tCommon.hChannel, sizeof(tRecvPkt), &tRecvPkt, ptApp->tCommon.ulTimeout)))
  {
    App_HandlePacket(ptApp, &tRecvPkt);
  }

  /* unregister from read / write notifications */
  CoE_UnegisterNotifications(ptCoEApp);

  /* unregister on EcatEsm task (indications for status) */
  App_SendRecvEmptyPkt(ptApp, RCX_UNREGISTER_APP_REQ);

  return tResult;
}


/*************************************************************************************************
 * @brief This method handles all packets which will be received by CoE application.
 * 
 * @param ptApp Pointer to application data.
 * @param ptPacket Pointer to CifX packet.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
void App_HandlePacket(APPLICATION_T*  ptApp, CIFX_PACKET* ptPacket)
{
  TLR_PACKET_HEADER_T* ptPck = (TLR_PACKET_HEADER_T*)ptPacket;

  /* convert to specific application */
  COE_APPLICATION_T* ptCoEApp = (COE_APPLICATION_T*)ptApp;


  switch(ptPck->ulCmd)
  {
    case ODV3_WRITE_OBJECT_IND:
      CoE_HandleWriteIndication(ptCoEApp, (ODV3_WRITE_OBJECT_IND_T*)ptPacket);
      break;

    case ODV3_READ_OBJECT_IND:
      CoE_HandleReadIndication(ptCoEApp, (ODV3_READ_OBJECT_IND_T*)ptPacket);
      break;

    case ECAT_ESM_ALCONTROL_CHANGED_IND:
      CoE_HandleAlControlChangedRequest(ptCoEApp, (ECAT_ESM_ALCONTROL_CHANGED_IND_T*)ptPacket);
      break;

    case ECAT_ESM_ALSTATUS_CHANGED_IND:
      CoE_HandleAlStatusChangedIndication(ptCoEApp, (ECAT_ESM_ALSTATUS_CHANGED_IND_T*)ptPacket);
      break;

    default:
      DEBUG("Unknown packet. ulCmd=0x%x\n", ptPacket->tHeader.ulCmd);
      break;
  }
}


/*************************************************************************************************
 * @brief This method handles the cyclic process data exchange.
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
void App_HandleProcessData(APPLICATION_T* ptApp)
{
  /* return value of function */
  TLR_RESULT tResult = TLR_S_OK;

  /* convert to specific application */
  COE_APPLICATION_T* ptCoEApp = (COE_APPLICATION_T*)ptApp;


  /* cyclic io handing happens here */
  /* read data */
  tResult = xChannelIORead(ptApp->tCommon.hChannel, 0, 0, ptApp->tCommon.ulReadBufferSize, ptApp->tCommon.pabReadBuffer, 2);

  /* the action depends on the return value we have received */
  if ((CIFX_DEV_EXCHANGE_FAILED == tResult) || (CIFX_DEV_EXCHANGE_TIMEOUT == tResult))
  {
    DEBUG("F");
  }
  else if (CIFX_DEV_NO_COM_FLAG == tResult)
  {
    DEBUG("F");
  }
  else if (CIFX_NO_ERROR == tResult)
  {
    DEBUG("X");
    
    /* We copy the memory we have read to the memory we want to send because we just want to mirror the data. */
    memcpy (ptApp->tCommon.pabWriteBuffer, ptApp->tCommon.pabReadBuffer, ptApp->tCommon.ulWriteBufferSize);     

    /* write data */
    tResult = xChannelIOWrite(ptApp->tCommon.hChannel, 0, 0, ptApp->tCommon.ulWriteBufferSize, ptApp->tCommon.pabWriteBuffer, 2);       
  }
  else
  {
    /* received unexpected failure */
    DEBUG("F");
  }
  
}
